/* getSunGlintAng.c */

/*******************************************************************
* getSunGlintAng()
*
* Description:
*             This function takes the pixel location, satllite location
*             and unit sun vector in ECR corrdinates  and calculates the sun glint 
*             angle and return the computed sun glint angle in degree.
*
* Input parameters:
*       double lat              -geodetic latitude (deg) of the pixel [-90,90]
*       double lon              -geodetic longitude (deg) of the pixel [-180,180]
*       double sclat            -geodetic latitude (deg) of the satellite [-90,90]
*       double sclon            -geodetic longitude (deg) of the satellite [-180,180]
*       double scalt            -altitude (Km) of the satellite
*       double sunVec           -unit sun vector in ECR coordinates
*
* Output parameters:
*       double zasun            -sun zenith angle in degree [0,180]
*       double zaview           -view zenith angle in degree [0,180]
*       double sun_view_az      -azimuth angle between the sun and view vector 
*                                in degree [-180,180]
*                                a value of -9999.0 indicates pixel not in view or
*                                sun not on pixel
*
* Return:
*       double sunGlintAng      -the sun glint angle (the seperation angle 
*                                between the sun vector and reflected view 
*                                vector) in degree [0,180]
*                                a value of -9999.0 indicates pixel not in view or
*                                sun not on pixel
* Routines called:
*       getVecECR()
*
*
* Note:
*
*       GCI, ECI, GEI           -Geocentric Equatorial Inertial Coordinates
*       ECR, ECEF               -Earth Centered Rotating Coordinates or 
*                                Earth Centered, Earth Fixed Coordinates  
* Log: 
*       5/26/2006        Joyce Chou          first create 
*
**********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "sunGlint.h"                 /* header file for sun glint package */ 
#define  RAD2DEG         57.29577951  /* 180 deg / pi radians */
#define  DEG2RAD         0.017453292  /* pi radians / 180 deg */
#define  MISSDOUBLE      -9999.0      /* missing value for double */
#define  BUGDOUBLE       -8888.0      /* bug value for double */



double getSunGlintAng(double lat, double lon, double sclat,double sclon,double scalt,
            double sunVec[3],
            double *zasun, double *zaview, double *sun_view_az)
{
    double sunGlintAng;   /* sun glint angle in deg */
    double satVec[3];     /* satellite position vector in ECR */
    double pixVec[3];     /* pixel position vector in ECR */
    double viewVec[3];    /* viewing vector in ECR        */ 
    double viewRefl[3];    /* reflected viewing vector in ECR */

    double Up[3];         /* up vector in ECR */
    double Cvec[3];       /* cross-view vector in ECR */
    double Avec[3];       /* along-view vector in ECR */
    double dotVpix;       /* the dot product of view and Up  vectors */
    double sunDotView;    /* the dot product of sun and view vectors */
    double sunDotA;       /* the dot product of sun and along-view vectors */
    double sunDotC;       /* the dot product of sun and cross-view vectors */
    double UpDotSun;      /* the dot product of Up and sun vector */

    double d;             /* the magnitude of a vector */ 
    double clon,clat;     /* cos values for pixel's lat/lon */ 
    double slon,slat;     /* sin values for pixel's lat/lon */ 
    double csclon,csclat; /* cos values for SAT sclat/sclon */ 
    double ssclon,ssclat; /* sin values for SAT sclat/sclon */ 



    /******************************************************************
    * convert pixel location to pixel position vector in ECR coordinate.
    ******************************************************************/
    clon=cos(lon*DEG2RAD);
    clat=cos(lat*DEG2RAD);
    slon=sin(lon*DEG2RAD);
    slat=sin(lat*DEG2RAD);
    getVecECR(clat,clon,slat,slon,0.0,pixVec);


    /*********************************************************
    * convert pixel location and satellite location to
    * position vectors in ECR coordinate.
    *********************************************************/
    csclon=cos(sclon*DEG2RAD);
    csclat=cos(sclat*DEG2RAD);
    ssclon=sin(sclon*DEG2RAD);
    ssclat=sin(sclat*DEG2RAD);
    getVecECR(csclat,csclon,ssclat,ssclon,scalt,satVec);
    
    /*****************************************************************
    * compute unit viewing vector in ECR.
    * viewVec[] = pixVec[] - satVec[]
    *****************************************************************/
    viewVec[0]=pixVec[0] - satVec[0];
    viewVec[1]=pixVec[1] - satVec[1];
    viewVec[2]=pixVec[2] - satVec[2];

    /* unitize the viewVec[] */
    d=sqrt(viewVec[0]*viewVec[0]+viewVec[1]*viewVec[1]+viewVec[2]*viewVec[2]);
    viewVec[0]= viewVec[0]/d;
    viewVec[1]= viewVec[1]/d;
    viewVec[2]= viewVec[2]/d;

    /**************************************************
    * Get the view and solar zenith angles and check 
    * if the pixel can be seen from the satellite, 
    * and if the Sun shines on the pixel (atmospheric refaction is ignored)
    **************************************************/
    /* compute the Local up vector at the pixel position. */
    Up[0] = clon*clat;
    Up[1] = slon*clat;
    Up[2] = slat;
    d=sqrt(Up[0]*Up[0]+Up[1]*Up[1]+Up[2]*Up[2]);
    Up[0]= Up[0]/d;
    Up[1]= Up[1]/d;
    Up[2]= Up[2]/d;

    dotVpix = Up[0]*viewVec[0]+Up[1]*viewVec[1]+Up[2]*viewVec[2];
    

    /* compute the zenith angles for the view and sun vectors */
    if(dotVpix <= 1.0 && dotVpix >= -1.0) 
       *zaview = RAD2DEG* acos((-1.0)*dotVpix);
    else
    {
       if(dotVpix > 1.0 && dotVpix < 1.001)
          *zaview = 180.0;
       else if(dotVpix > (-1.001) && dotVpix < (-1.0))
          *zaview = 0.0; 
       else
          *zaview = BUGDOUBLE;
    }

        
    UpDotSun = Up[0]*sunVec[0]+Up[1]*sunVec[1]+Up[2]*sunVec[2];
  
/*  printf("sunVec: %f %f %f\n",sunVec[0],sunVec[1],sunVec[2]);
    printf("pixVec: %f %f %f\n",pixVec[0],pixVec[1],pixVec[2]);
    printf("satVec: %f %f %f\n",satVec[0],satVec[1],satVec[2]);
    printf("Up: %f %f %f\n",Up[0],Up[1],Up[2]);
    printf("UpDotSun= %f\n",UpDotSun);
 */
    
    if(UpDotSun <= 1.0 && UpDotSun >= -1.0)
       *zasun = RAD2DEG* acos(UpDotSun);
    else
    {
       if(UpDotSun > 1.0 && UpDotSun < 1.001)
          *zasun = 0.0;
       else if (UpDotSun > (-1.001) && UpDotSun < (-1.0))
          *zasun = 180.0;
       else
          *zasun = BUGDOUBLE;  
    }
    
    if( (*zaview) < 90.0 )   /* If pixel in view  (else, will enter flag values) */
    {
       if( (*zasun) < 90.0 )   /* If sun shines on pixel (else, will enter flag values) */ 
       {
          /*********************************************************************
           get relected view in ECR/ECEF and get anglular separation from Sun vector
          **********************************************************************/ 
          viewRefl[0] = viewVec[0] - 2*dotVpix*Up[0];
          viewRefl[1] = viewVec[1] - 2*dotVpix*Up[1];
          viewRefl[2] = viewVec[2] - 2*dotVpix*Up[2];
          d=sqrt(viewRefl[0]*viewRefl[0]+viewRefl[1]*viewRefl[1]+viewRefl[2]*viewRefl[2]);
          viewRefl[0]=viewRefl[0]/d;
          viewRefl[1]=viewRefl[1]/d;
          viewRefl[2]=viewRefl[2]/d;

          sunDotView = sunVec[0]*viewRefl[0]+sunVec[1]*viewRefl[1]+sunVec[2]*viewRefl[2];
          if(sunDotView <= 1.0 && sunDotView >= -1.0)
             sunGlintAng = RAD2DEG*acos(sunDotView);
          else
          {
             if(sunDotView > 1.0 && sunDotView < 1.001)
                sunGlintAng = 0.0;
             else if (sunDotView > (-1.001) && sunDotView < (-1.0))
                sunGlintAng = 180.0;
             else
                sunGlintAng = BUGDOUBLE;
          }
      
          /*******************************************
           get sun_view_azim angle
          **************************************/
          /* get corss-view vector: Cvec = viewVec X Up */
          Cvec[0]=viewVec[1]*Up[2]-viewVec[2]*Up[1];
          Cvec[1]=viewVec[2]*Up[0]-viewVec[0]*Up[2];
          Cvec[2]=viewVec[0]*Up[1]-viewVec[1]*Up[0];
          d=sqrt(Cvec[0]*Cvec[0]+Cvec[1]*Cvec[1]+Cvec[2]*Cvec[2]);

          if(d > 0.000001) /* if view is not straight down */  
          {
             /* unitize Cvec[] */
             Cvec[0] =Cvec[0]/d;
             Cvec[1] =Cvec[1]/d;
             Cvec[2] =Cvec[2]/d;

             /* get along-view vector: Avec = Up X Cvec */
             Avec[0]=Up[1]*Cvec[2]-Up[2]*Cvec[1];
             Avec[1]=Up[2]*Cvec[0]-Up[0]*Cvec[2];
             Avec[2]=Up[0]*Cvec[1]-Up[1]*Cvec[0];
             d=sqrt(Avec[0]*Avec[0]+Avec[1]*Avec[1]+Avec[2]*Avec[2]);
             Avec[0] =Avec[0]/d;
             Avec[1] =Avec[1]/d;
             Avec[2] =Avec[2]/d;

             /* compute sun_view_azim angle */
             sunDotA=sunVec[0]*Avec[0]+sunVec[1]*Avec[1]+sunVec[2]*Avec[2];
             sunDotC=sunVec[0]*Cvec[0]+sunVec[1]*Cvec[1]+sunVec[2]*Cvec[2];

           /*
             printf(" sunVec[]= %f %f %f\n",sunVec[0],sunVec[1],sunVec[2]);
             printf(" Cvec[]= %f %f %f\n",Cvec[0],Cvec[1],Cvec[2]);
             printf(" Avec[]= %f %f %f\n",Avec[0],Avec[1],Avec[2]);
             printf(" sunDotA=%f \n",sunDotA);
             printf(" sunDotC=%f \n",sunDotC);
            */
             *sun_view_az = RAD2DEG*atan2((-1.0)*sunDotC, sunDotA);

          }
          else  /* if satellite look straight down */
             *sun_view_az = 0.0;
       }
       else /* sun not on pixel */
       {
          sunGlintAng  = MISSDOUBLE;
          *sun_view_az = MISSDOUBLE;
       }
    }
    else /* pixel not in view */
    {

       sunGlintAng  = MISSDOUBLE;
       *sun_view_az = MISSDOUBLE;
    }
    
    return(sunGlintAng);
}
