/* getSunVecECR.c */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "sunGlint.h"

/*******************************************************************
* getSunVecECR()
*
* Description:
*   This function computes the unit sun position vector in ECR 
*   for the given time.
*
* Input parameters:
*   year          int     - 4 digit year
*   doy           int     - day of year (1-366)
*   secs_of_day   float   - seconds of the day including fractional part 
*   GHA           double  - Greenwich hour angle (rad)
*
* Output:
*   sunVec[3]     double  - unit sun position vector in ECR 
*
* Functions called:
*   sun2000SG()
*   convGCI2ECR() 
*
* Return:
*   0
*
* Log:
*   5/25/2006        Joyce Chou        first created
* 
*******************************************************************/
int getSunVecECR(int year,int doy,float secs_of_day, double GHA, double sunVec[3])
{

    double  rs;            /* solar distance (AU) */
    double  mag;           /* magnitude */

    /* get sunVec[] in GCI coordinate. */
    sun2000SG((long)year,(long)doy,(double)secs_of_day,sunVec,&rs);
    /* convert sunVec[] from GCI to ECR coordinate */
    convGCI2ECR(sunVec,GHA);
   
    /* unitize the vector */
    mag=sqrt(sunVec[0]*sunVec[0]+sunVec[1]*sunVec[1]+sunVec[2]*sunVec[2]);
    sunVec[0]=sunVec[0]/mag;
    sunVec[1]=sunVec[1]/mag;
    sunVec[2]=sunVec[2]/mag;
   
    return 0;
}


/*******************************************************************
* convGCI2ECR()
*
* Description:
*   This function converts a vector from GCI/ECI coordinates 
*   to ECR/ECEF coordinates 
*
* Input parameters:
*   Vec[3]        double  - the vector in GCI 
*   GHA           double  - Greenwich hour angle (rad)
*
* Output:
*   Vec[3]        double  - the vector in ECR 
*
* Functions called:
*
* Return:
*   0
*
* Log:
*   5/25/2006        Joyce Chou        first created
*   7/24/2006        Joyce Chou        fix a bug in the rotation calculation
* 
*******************************************************************/
int convGCI2ECR(double Vec[3], double GHA)
{

    double cGHA;          /* cos(GHA) */
    double sGHA;          /* sin(GHA) */
    double VEC[3];

    cGHA=cos(GHA);
    sGHA=sin(GHA);

    /* convert from GCI to ECR coordinates by applying a      */
    /* rotation angle of GHA.                                  */
    /* note the Z component (satVec[2])  does not change       */
    VEC[0] =Vec[0]*cGHA+Vec[1]*sGHA;
    VEC[1] =(-1.0)*Vec[0]*sGHA+Vec[1]*cGHA;

    /* assign the vector back to Vec[] */
    Vec[0]=VEC[0];
    Vec[1]=VEC[1];

    return 0;
}

