/* getVecECR.c */

/*******************************************************************************
*
*  getVecECR()
*
*  Description:
*    Compute the position vector in ECR coordinates.
*    
*  Input parameters:
*    clat         double     cos(Geodetic latitude)
*    clon         double     cos(Geodetic longitude)
*    slat         double     sin(Geodetic latitude)
*    slon         double     sin(Geodetic longitude)
*    alt          double     Altitude (Km)
*
*  Output:
*    ECRvec[3]    double     position vector in ECR coordinates
*
*  Return:
*    0
*
* Log:
*   5/25/2006        Joyce Chou        first created
*
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define DEG2RAD       0.017453292    /* pi radians / 180 deg */
#define R_EQUATOR     6378.137       /* Radius of the Earth equator in km   */
#define FLATC         0.00335281     /* Earth's oblateness coefficient      */
#define lEcc2         0.0066943787   /* lEcc2= 2.0 * FLATC - FLATC * FLATC */


int getVecECR (
	double clat,	        /* cos(Geodetic latitude)  */ 
	double clon,	        /* cos(Geodetic longitude) */
        double slat,            /* sin(Geodetic latitude)  */
        double slon,            /* sin(Geodetic longitude) */
	double alt,	        /* Altitude (Km)                      */
	double ECRvec[3])	/* position vector in ECR coordinates */
{
   double up[3];
   static double Z;	        /* Z components of the position vector */
   static double XY;		/* XY plane magnitude of the position vector */
   double epsilon, nc;		/* Variables to hold intermediate results    */

      
      
      epsilon = 1.0 / sqrt(1.0 - lEcc2 * slat * slat);
      nc = epsilon * R_EQUATOR;
      

      /*****************************************************
      *  Calculate the Z component of the position vector  *
      *****************************************************/
      
      Z = nc * (1.0 - lEcc2) * slat;
      
      /*************************************************
      *  Calculate the XY plane magnitude of the Gvec  *
      *************************************************/
      
      XY = nc * clat;


      /****************************************************
      *  Calculate the X component of the position vector  *
      ****************************************************/
      
      ECRvec[0] = XY * clon; 
      ECRvec[1] = XY * slon; 
      ECRvec[2] = Z; 

      if(alt > 0.00001) /* if there is a altitude */
      {

        up[0] = clon*clat;
        up[1] = slon*clat;
        up[2] = slat;
        ECRvec[0] =ECRvec[0] +up[0]*alt;
        ECRvec[1] =ECRvec[1] +up[1]*alt;
        ECRvec[2] =ECRvec[2] +up[2]*alt;
      }
      
   return 0;
}
