/* sunGlintAngle.c */

/*******************************************************************
* sunGlintAngle()
*
* Description:
*             This function takes the pixel location, satllite location
*             and time information and calculates the sun glint 
*             angle and return the computed sun glint angle in degree.
*
* Input parameters:
*       double lat              -geodetic latitude (deg) of the pixel [-90, 90] 
*       double lon              -geodetic longitude (deg) of the pixel [-180,180]
*       double sclat            -geodetic latitude (deg) of the satellite [-90, 90]
*       double sclon            -geodetic longitude (deg) of the satellite [-180,180]
*       double scalt            -altitude (Km) of the satellite
*       int    year             -4 digit year
*       int    month            -month  [1,12]
*       int    day              -day of the month [1,31]
*       int    hour             -hour [0,23]
*       int    min              -minute [0,59]
*       float  fsec             -second including fractional part [0,59.999]
*
* Output parameters:
*       double zasun            -sun zenith angle in degree [0, 180]
*       double zaview           -view zenith angle in degree [0, 180]
*       double sun_view_azim    -azimuth angle between the sun and view 
*                                vector in degree [-180, 180]
*                                a value of -9999.0 indicates pixel not in view or
*                                sun not on pixel
*
* Return:
*       double sunGlintAng      -the sun glint angle (the seperation angle 
*                                between the sun vector and reflected view 
*                                vector) in degree [0, 180]
*                                a value of -9999.0 indicates pixel not in view or
*                                sun not on pixel
* Routines called:
*       getDOY() 
*       gha2000SG()
*       getSunVecECR()
*       getSunGlintAng() 
*
* Note :
*
*       GCI, ECI, GEI           -Geocentric Equatorial Inertial Coordinates
*       ECR, ECEF               -Earth Centered Rotating Coordinates or 
*                                Earth Centered, Earth Fixed Coordinates  
*
* Log:
*       5/26/2006          Joyce Chou      first created
**********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "sunGlint.h"

double sunGlintAngle(double lat, double lon, double sclat,double sclon,double scalt,
            int year, int month, int day, int hour, int min,float fsec,
            double *zasun, double *zaview, double *sun_view_azim)
{


    double sunVec[3];     /* sun vector in ECR            */
    double sunGlintAng;   /* sun glint angle in deg */
    float  secs_of_day;   /* seconds of the day as a float */
    int    doy;           /*  day of the year */
    double fdoy;          /* fractional day of the year as a float */
    double GHA;           /* Greenwich hour angle (rad) */


    /*******************************************************************
    * get the unit sun position vector in ECR for the given time 
    *******************************************************************/
    /* get the day of year */
    doy=getDOY(year,month,day,hour,min,fsec,&secs_of_day);
    /* get the fractional day of year */
    fdoy=(double)doy+secs_of_day/86400.0;
    /* get the Greenwich hour angle */ 
    gha2000SG((long)year,fdoy,&GHA);

    /* get the unit sun vector in ERC */
    getSunVecECR(year,doy,secs_of_day,GHA,sunVec);


    /*******************************************************************
    * get the sunglint angle which is the separation angle between
    * the reflected view and Sun vector. 
    * also compute the side products: zaview, zasun and sun_view_az
    *********************************************************************/

    sunGlintAng=getSunGlintAng(lat,lon,sclat,sclon,scalt,sunVec,zasun,zaview,sun_view_azim);

    return(sunGlintAng);

}
