/***************************************************************
* util.c   
* This file contains some utility functions used to compute 
* the sun glint angle.
***************************************************************/

#include <stdio.h>
#include "math.h"
#include "sunGlint.h"
/***************************************************************
* date2jdaySG() 
*
* Description:
*   Function to convert calendar date to julian day (day of year).
*
* Input:
*   year   4 digit year    
*   month  month of the year
*   day    day of the month
*
* Return:
*   int   julian day   (day of year)
*
* Log:
*   5/25/2006   Joyce Chou    first created
*
****************************************************************/
int date2jdaySG(int year,int month,int day)
{
  int jld;
  int accumDays[13]={0,31,59,90,120,151,181,212,243,273,304,334,365};
  int leapAccumDays[13]={0,31,60,91,121,152,182,213,244,274,305,335,366};

  if((year%4 == 0 && year%100 != 0) || year%400 == 0) /* leap year */
     jld=leapAccumDays[month-1]+day;
  else
     jld=accumDays[month-1]+day;
  return jld;
}


/***************************************************************
* getDOY() 
* 
* Description:
*   Function to convert calendar date to julian day (day of year).
*   also get the seconds of the day.
*
* Input:
*   year   int    - 4 digit year   
*   month  int    - month of the year
*   day    int    - day of the month
*   hour   int    - hour 
*   min    int    - minute
*   fsec   float  - second in float 
*
* Output:
*   secs_of_day  float   - seconds of the day
*
* Return:
*   int   doy   int  - julian day   (day of year)
*
* Log:
*   5/25/2006   Joyce Chou    first created
*
****************************************************************/

int getDOY(int year,int month,int day,int hour,int min,float fsec,float *secs_of_day)
{
   int doy;
   doy=date2jdaySG(year,month,day);
   
   /* also compute the seconds of the day */
   *secs_of_day = hour*3600.0 + min*60.0 + fsec; 
   

   return doy;
}



/****************************************************************************/
/****************************************************************************/
/* NOTE            NOTE            NOTE              NOTE                   */
/* The following functions are taken directly from the GEOLOCATION package. */
/* But the names have been changed to avoid double definition               */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/


#define  RAD2DEG         57.29577951
#define  DEG2RAD         0.017453292  /* pi radians / 180 deg */

/***************************************************************************
 *
 *  int gha2000SG(iyr,day gha) 
 *                
 *  Description:
 *  This subroutine computes the Greenwich hour angle in degrees for the
 *  input time. It uses the model referenced in the Astronomical Almanac
 *  for 1984, Section S (Supplement) and documented for the SeaWiFS  
 *  Project in "Constants and Parameters for SeaWiFS Mission Operations",
 *  in TBD.  It includes the correction to mean sideral time for nutation
 *  as well as precession.
 *
 *  This routine originally coded in FORTRAN by Fred Patt, GSC, Laurel, MD.
 *                
 *  Input Parameters:
 *  iyr       long       year (four digits)
 *  day       double     day (time of day as fraction)
 * 
 *  Output Parameters:
 *  gha       double     Greenwich hour angle (rad)
 *                                                           
 *                                                          
 *  Routines called:                                       
 *  jdSG        computes day of year from calendar date                 
 *  ephparmSG   computes mean solar longitude and anomaly and
 *              mean lunar longitude and ascending node
 *  nutateSG    compute nutation corrections to longitude and obliquity
 *
 *
 **************************************************************************
*/

int gha2000SG(long iyr,double day,double *gha)
 
{

long imon=1;
static long nutime=-99999;
double fday,ip,t,gmst,nt;
double xls,gs,xlm,omega,dpsi,eps;
long iday,jday;

/* Compute days since J2000 */
fday=modf(day,&ip);
iday=(long)ip;
jday=jdSG(iyr,imon,iday);
t=jday-2451545.50+fday;

/* Compute Greenwich Mean Sidereal Time (degrees) */
gmst=100.46061840 + 0.98564736630*t + 2.908e-13 *t*t;
/* Check if need to compute nutation correction for this day */
nt=t;
if (nt!=nutime)     
   {          
   nutime=nt;
   ephparmsSG(t,&xls,&gs,&xlm,&omega);
   nutateSG(t,xls,gs,xlm,omega,&dpsi,&eps);
   }

/* include apparent time correction and time-of-day */
*gha = gmst + dpsi*cos(eps/RAD2DEG) +fday*360.0;
*gha = fmod(*gha,360.0); 

  
if (*gha<0.0) *gha+=360.0;

/* convert gha from deg to rad */
*gha = (*gha)*DEG2RAD;

return 0;

}
/***************************************************************************
 *
 *  sun2000SG(iyr,iday,sec,sun,*rs) 
 *                
 *  Description:
 *  This subroutine computes the sun vector in geocentric inertial 
 *  (equatorial) coordinates. It uses the model referenced in The
 *  Astonomical Almanac for 1984, Section S (Supplement) and
 *  documented for the SeaWiFS Project in "Constants and Parameters 
 *  for SeaWiFS Mission Operations", in TBD. The accuracy of the Sun
 *  vector is approximately 0.1 arcminute. This routine originally
 *  coded in FORTRAN by Fred Patt, GSC, Laurel, MD.
 *                
 *  Input Parameters:
 *  iyr     long     year, four digits (i.e., 1993)
 *  iday    long     day of year (1-366)
 *  sec     double   seconds of day
 *
 *  Output Parameters:
 *  sun[3]  double    unit sun vector in geocentric inertial
 *                   coordinates of date
 *  rs      double    magnitude of the sun vector (AU)
 *                                                           
 *                                                          
 *  Routines called:                                       
 *  jdSG             computes Julian day from calendar date
 *  ephparmsSG       computes mean solar longitude and anomaly and
 *                   mean lunar longitude and ascending node
 *  nutateSG         compute nutation corrections to longitude
 *                   and obliquity
 *                                                        
 *  
 * $Log: sun2000.c,v $
 * Revision 1.4  1997/08/04 16:36:14  bruce
 * added return status variable
 *
 * Revision 1.3  1997/08/04 15:17:01  bruce
 * remove unreferenced variables
 *
 * Revision 1.2  1996/10/02  17:29:08  udaya
 * Included GEOLOCATE.h header file (for function prototypes).
 *
 * Revision 1.1  1996/01/22  22:07:22  udaya
 * adding file first time
 *
 * Revision 2.0  1993/05/13  17:12:51  pathfps
 * Set to 2.0
 *
 * Revision 1.4  1993/04/29  19:50:19  pathfps
 * Added protos.h  A. Sigmund, SAIC/GSC
 *
 * Revision 1.3  1993/02/11  16:30:24  pathfps
 * deleted print statement
 *
 * Revision 1.2  1993/02/11  14:16:57  pathfps
 * Removed prototype header. B. Mack, HSTX
 *
 * Revision 1.1  1993/02/10  18:08:15  pathfps
 * Original Development by A. Sigmund, SAIC/GSC
 *                                               
 *
 **************************************************************************
*/

int sun2000SG(long iyr,long iday,double sec,double sun[3],double *rs) 

{
static double xk=0.0056932;
static long imon=1;
double dls,t,xls,gs,xlm,omega,dpsi,eps,nt,g2,g4,g5;
double xlsg,xlsa;
static int nutime=-99999;
double sgs, cgs, sg2, cg2, sg5, cg5;          /* BBD added   7/18/94  */

/* Compute floating point days since Jan 1.5, 2000. */
/* Note that the Julian day starts at noon on the specified date */
t=jdSG(iyr,imon,iday)-2451545.00+(sec-43200.0)/86400.0;

/* Compute solar ephemeris parameters */
ephparmsSG(t,&xls,&gs,&xlm,&omega);

/* Check if need to compute nutation corrections for this day */
nt=t;
if (nt!=nutime)
    {
    nutime=nt;
    nutateSG(t,xls,gs,xlm,omega,&dpsi,&eps);
    /*printf("t,xls,gs,xlm,omega,dpsi,eps:%lf %lf %lf %lf %lf %lf %lf\n",t,xls,gs,xlm,omega,dpsi,eps);*/
    }

/* Compute planet mean anomalies. */
/* Venus mean anomaly */
g2=50.40828+1.60213022*t;
g2=fmod(g2,360.0);

/* Mars mean anomaly */
g4=19.38816+0.52402078*t;
g4=fmod(g4,360.0);

/* Jupiter mean anomaly */
g5=20.35116+0.08309121*t;
g5=fmod(g5,360.0);

sgs=(float)(sin(gs/RAD2DEG));                    /* BBD Added  */
cgs=(float)(cos(gs/RAD2DEG));                    /* 7/18/94    */
sg2=(float)(sin(g2/RAD2DEG));
cg2=(float)(cos(g2/RAD2DEG));
sg5=(float)(sin(g5/RAD2DEG));
cg5=(float)(cos(g5/RAD2DEG));

/* Compute solar distance (AU) */
/* *rs=(float)(1.00014-0.01671*cos(gs/RAD2DEG)-0.00014*cos(2.0*gs/RAD2DEG));  */

*rs = (double)(1.00014-0.01671*cgs - 0.00014*(1.-2 * sgs * sgs));   /* BBD added
                                                                   7/18/94  */
/* Compute geometric solar longitude */
dls=(6893.-4.6543463-4*t)*sgs 
    +72.*2*sgs*cgs
    -7*(cgs*cg5+sgs*sg5)
    +6.*sin((xlm-xls)/RAD2DEG)
    +5.*sin((4.*gs-8.*g4+3.*g5)/RAD2DEG)
    -5.* (1-2*(sgs*cg2-cgs*sg2)*(sgs*cg2-cgs*sg2))
    -4.*(sgs*cg2-cgs*sg2)
    +4.*cos((4.*gs-8.*g4+3.*g5)/RAD2DEG)
    +3.*2*(sgs*cg2-cgs*sg2)*(cgs*cg2+sgs*sg2)
    -3.*sg5
    -3.*2*(sgs*cg5-cgs*sg5)*(cgs*cg5+sgs*sg5);   /* arcseconds */

xlsg=xls+dls/3600.0;

/* Compute apparent solar longitude; includes corrections for  */ 
/* nutation in longitude and velocity aberration.              */
xlsa=xlsg+dpsi-xk/(*rs);

/* Compute unit sun vector */

sun[0]=(double)(cos(xlsa/RAD2DEG));
sun[1]=(double)(sin(xlsa/RAD2DEG)*cos(eps/RAD2DEG));
sun[2]=(double)(sin(xlsa/RAD2DEG)*sin(eps/RAD2DEG));
 
return 0;

}

/***************************************************************************
 *
 *  int jdSG(i,j,k) 
 *                
 *  Description:
 *  This function converts a calendar date to the corresponding Julian
 *  day starting at noon on the calendar date. The algorithm used is
 *  from Van Flandern and Pulkkinen, Ap. J. Supplement Series 41,
 *  November 1079, p.400.
 *                
 *  Input Parameters:
 *  i     long          year-e.g. 1970
 *  j     long          month (1-12)
 *  k     long          day (1-31)
 *
 *  Output Parameters:
 *  jd    long          Julian day    
 *                                                           
 *  Called By:     sun2000SG 
 *                                                          
 *  Routines called:                                       
 *  none                                                        
 *
 *  Written by Frederick S. Patt, GSC, 11/4/92.
 *  Converted from FORTRAN to C by A.Sigmund, GSC, 2/14/92.
 *                                                      
 * $Log: jd.c,v $
 * Revision 1.2  1997/08/04 15:13:49  bruce
 * remove unreferenced variables
 *
 * Revision 1.1  1996/01/22  22:07:17  udaya
 * adding file first time
 *
 * Revision 2.0  1993/05/13  17:12:33  pathfps
 * Set to 2.0
 *
 * Revision 1.4  1993/04/29  19:46:19  pathfps
 * Added protos.h  A. Sigmund, SAIC/GSC
 *
 * Revision 1.3  1993/02/22  22:09:17  pathfps
 * deleted addition of 1900, A. Sigmund, SAIC/GSC
 *
 * Revision 1.2  1993/02/11  14:11:47  pathfps
 * Removed prototype header. B. Mack, HSTX
 *
 * Revision 1.1  1993/02/10  18:05:19  pathfps
 * Original Development by A. Sigmund, SAIC/GSC
 *                                               
 *
 **************************************************************************
*/

long jdSG(long i,long j,long k)
{
   long jdd;


   jdd=367*i - 7*(i+(j+9)/12)/4 + 275*j/9 + k +1721014;

   /* This additional calculation is needed only for dates outside of */
   /* the period March 1, 1900 to February 28, 2100.                  */

   /* jdd=jdd + 15 - 3*((i+(j-9)/7)/100+1)/4;                           */

  
   return jdd;

}

/***************************************************************************
 *
 *  nutateSG(t,xls,gs,xlm,omega,dpsi,eps) 
 *                
 *  Description:
 *  This subroutine computes the nutation in longitude and the obliquity
 *  of the ecliptic corrected for nutation.  It uses the model referenced
 *  in The Astronomical Alkmanac for 1984, Section S (Supplement) and
 *  documented for the SeaWiFS Project in "Constants and Parameters for
 *  SeaWiFS Mission Operations", in TBD.  These parameters are used to
 *  compute the apparent time correction to the Greenwich Hour Angle and
 *  for the calculation of the geocentric Sun vector.  The input
 *  ephemeris parameters are computed using subroutine ephparms.  Terms
 *  are included to 01 arcsecond.
 *                
 *  Input Parameters:
 *  t     double  time indays since January 1, 2000 at 12 hours UT
 *  xls   double  mean solar longitude (degrees)
 *  gs    double  mean solar anomaly (degrees)
 *  xlm   double  mean lunar longitude (degrees)
 *  omega double  ascending node of mean lunar orbit (degrees)
 *
 *  Output Parameters:
 *  dpsi  double  nutation in longitude (degrees)
 *  eps   double  obliquity of the ecliptic (degrees)
 *                                                           
 *  Called By:     sun2000SG,gha2000SG 
 *                                                          
 *  Routines called: None.
 *
 *  Created by Frederick S. Patt, GSC, 10/21/92.
 *  Converted from FORTRAN to C by A.Sigmund, GSC, 2/14/93.
 *                                                      
 * $Log: nutate.c,v $
 * Revision 1.2  1997/08/04 15:11:21  bruce
 * remove unreferenced variables
 *
 * Revision 1.1  1996/01/22  22:07:19  udaya
 * adding file first time
 *
 * Revision 2.0  1993/05/13  17:12:44  pathfps
 * Set to 2.0
 *
 * Revision 1.4  1993/04/29  19:49:07  pathfps
 * Added protos.h  A. Sigmund, SAIC/GSC
 *
 * Revision 1.3  1993/02/22  22:06:08  pathfps
 * Changed -7 to e-7, A. Sigmund, SAIC/GSC
 *
 * Revision 1.2  1993/02/11  14:15:54  pathfps
 * Removed prototype header. B. Mack, HSTX
 *
 * Revision 1.1  1993/02/10  18:07:11  pathfps
 * Original Development by A. Sigmund, SAIC/GSC
 *                                               
 *
 **************************************************************************
*/
int nutateSG(double t,double xls,double gs,double xlm,double omega,
           double *dpsi,double *eps)

{

    double deps,epsm;

    /* Nutation in longitude */
    *dpsi= -17.1996*sin(omega/RAD2DEG)
           +0.2062*sin(2.*omega/RAD2DEG)
           -1.3187*sin(2.*xls/RAD2DEG)
           +0.1426*sin(gs/RAD2DEG)
           -0.2274*sin(2.*xlm/RAD2DEG);

    /* Mean obliquity of the Ecliptic */
    epsm=23.4392910 - 3.560e-7 *t;

    /* Nutation in obliquity */
    deps=9.2025*cos(omega/RAD2DEG)+0.5736*cos(2.*xls/RAD2DEG);

    /* True obliquity of the ecliptic */
    *eps=epsm+deps/3600.0;

    *dpsi=*dpsi/3600.0;

    return 0;

}
/***************************************************************************
 *
 *  int ephparmsSG(t,xls,gs,xlm,omega) 
 *                
 *  Description:
 *  This subroutine computes ephemeris parameters used by other Mission
 *  Operations routines: the solar mean longitude and mean anomaly, and
 *  the lunar mean longitude and mean ascending node.  It uses the model
 *  referenced in The Astronomical Almanac for 1984, Section S (Supplement)
 *  and documented for the SeaWiFS Project in "Constants and Parameters for
 *  SeaWiFS Mission Operations", in TBD.  These parameters are used to 
 *  compute the solar longitude and the nutation in longitude and obliquity.
 *                
 *  Input Parameters:
 *  t       double   time in days since January 1, 2000 at 12 hours UT
 *
 *  Output Parameters:
 *  xls     double   mean solar longitude (degrees)
 *  gs      double   mean solar anomaly (degrees)
 *  xlm     double   mean lunar longitude (degrees)
 *  omega   double   ascending node of mean lunar orbit (degrees)  
 *
 *                                                           
 *  Called By:      sun2000SG,gha2000SG 
 *
 *  Routines called: None.
 *                                                          
 *  Program written by Frederick S. Patt, GSC, 11/2/92.
 *  Converted form FORTRAN to C by A.Sigmund, GSC, 2/14/93.
 *                                                      
 * $Log: ephparms.c,v $
 * Revision 1.2  1997/08/04 15:12:59  bruce
 * remove unreferenced variables
 *
 * Revision 1.1  1996/01/22  22:07:11  udaya
 * adding file first time
 *
 * Revision 2.0  1993/05/13  17:12:25  pathfps
 * Set to 2.0
 *
 * Revision 1.3  1993/04/29  19:44:51  pathfps
 * Added protos.h  A. Sigmund, SAIC/GSC
 *
 * Revision 1.2  1993/02/11  14:09:10  pathfps
 * Removed prototype header. B. Mack, HSTX
 *
 * Revision 1.1  1993/02/10  18:02:47  pathfps
 * Original Development by A. Sigmund, SAIC/GSC
 *                                               
 *
 **************************************************************************
*/
int ephparmsSG(double t, double *xls,double *gs,double *xlm,double *omega)
{

    /* sun mean longitude */
    *xls=280.465920 + 0.98564735160*t;
    *xls=fmod(*xls,360.0);

    /* sun mean anomaly */
    *gs=357.527720 + 0.98560028310*t;
    *gs=fmod(*gs,360.0);

    /* moon mean longitude */
    *xlm=218.316430+13.176396480*t;
    *xlm=fmod(*xlm,360.0);

    /* ascending node of moon's mean orbit */
    *omega=125.044520 - 0.05295376480*t;
    *omega=fmod(*omega,360.0);

    return 0;

}
