C*
C*----------------------------------------------------------------------------
C* FUNCTION getSunGlintAng()
C*
C* Description:
C*   This function takes the pixel location, satllite location
C*   and unit sun vector in ECR corrdinates  and calculates the sun glint
C*   angle and return the computed sun glint angle in degree.
C*
C* Input parameters:
C*   double lat        -geodetic latitude (deg) of the pixel [-90,90]
C*   double lon        -geodetic longitude (deg) of the pixel [-180,180]
C*   double sclat      -geodetic latitude (deg) of the satellite [-90,90]
C*   double sclon      -geodetic longitude (deg) of the satellite [-180,180]
C*   double scalt      -altitude (Km) of the satellite
C*   double sunVec     -unit sun vector in ECR coordinates
C*
C* Output parameters:
C*   double zasun        -sun zenith angle in degree [0,180]
C*   double zaview       -view zenith angle in degree [0,180]
C*   double sun_view_az  -azimuth angle between the sun and view vector
C*                        in degree [-180,180]
C*                        a value of -9999.0 indicates pixel not in view or
C*                        sun not on pixel
C*
C* Return:
C*   double sunGlintAng  -the sun glint angle (the seperation angle
C*                        between the sun vector and reflected view
C*                        vector) in degree [0,180]
C*                        a value of -9999.0 indicates pixel not in view or
C*                        sun not on pixel
C* Routines called:
C*       getVecECR()
C*
C*
C* Note:
C*
C*       GCI, ECI, GEI      -Geocentric Equatorial Inertial Coordinates
C*       ECR, ECEF          -Earth Centered Rotating Coordinates or
C*                           Earth Centered, Earth Fixed Coordinates
C* Log:
C*       10/26/2006        Joyce Chou          first create
C*
C*----------------------------------------------------------------------------
C23456789012345678901234567890123456789012345678901234567890123456789012

      double precision FUNCTION getSunGlintAng(lat,lon,sclat,sclon,scalt
     &,sunVec,zasun,zaview,sun_view_az)

      implicit none

C-----input parameters
      double precision lat,lon,sclat,sclon,scalt
      double precision sunVec(3)

C-----output parameters
      double precision zasun,zaview,sun_view_az 
      double precision sunGlintAng

C-----constant variables
      double precision RAD2DEG,DEG2RAD
      parameter (RAD2DEG=57.29577951,DEG2RAD=0.017453292)
      double precision MISSDOUBLE,BUGDOUBLE
      parameter (MISSDOUBLE=-9999.0,BUGDOUBLE=-8888.0)

C-----satellite position vector in ECR
      double precision satVec(3)
C-----pixel position vector in ECR
      double precision pixVec(3)
C-----viewing vector in ECR 
      double precision viewVec(3)
C-----reflected viewing vector in ECR 
      double precision viewRefl(3)
C-----up vector in ECR
      double precision Up(3)
C-----cross-view vector in ECR
      double precision Cvec(3)
C-----along-view vector in ECR
      double precision Avec(3)

C-----variables for the dot products
      double precision dotVpix,sunDotView,sunDotA,sunDotC,UpDotSun


      double precision d,clon,clat,slon,slat
      double precision csclon,csclat,ssclon,ssclat
      double precision zero
      DATA zero /0.0/

C----------------------------------------------------------------------------
C-----convert pixel location to pixel position vector in ECR coordinate.
C----------------------------------------------------------------------------

      clon=cos(lon*DEG2RAD)
      clat=cos(lat*DEG2RAD)
      slon=sin(lon*DEG2RAD)
      slat=sin(lat*DEG2RAD)
      CALL getVecECR(clat,clon,slat,slon,zero,pixVec)


C----------------------------------------------------------------------------
C---  convert pixel location and satellite location to
C---  position vectors in ECR coordinate.
C----------------------------------------------------------------------------

      csclon=cos(sclon*DEG2RAD)
      csclat=cos(sclat*DEG2RAD)
      ssclon=sin(sclon*DEG2RAD)
      ssclat=sin(sclat*DEG2RAD)
      CALL getVecECR(csclat,csclon,ssclat,ssclon,scalt,satVec)

C----------------------------------------------------------------------------
C---  compute unit viewing vector in ECR.
C----------------------------------------------------------------------------

      viewVec(1)=pixVec(1) - satVec(1)
      viewVec(2)=pixVec(2) - satVec(2)
      viewVec(3)=pixVec(3) - satVec(3)

C---  unitize the viewVec() 
      d=sqrt(viewVec(1)*viewVec(1)+viewVec(2)*viewVec(2)+viewVec(3)*view
     &Vec(3))
      viewVec(1)= viewVec(1)/d
      viewVec(2)= viewVec(2)/d
      viewVec(3)= viewVec(3)/d
      

C----------------------------------------------------------------------------
C---  Get the view and solar zenith angles and check
C---  if the pixel can be seen from the satellite,
C---  and if the Sun shines on the pixel (atmospheric reflection 
C---  is ignored)
C----------------------------------------------------------------------------

C---  compute the Local up vector at the pixel position. 
      Up(1) = clon*clat
      Up(2) = slon*clat
      Up(3) = slat
      d=sqrt(Up(1)*Up(1)+Up(2)*Up(2)+Up(3)*Up(3))
      Up(1)= Up(1)/d
      Up(2)= Up(2)/d
      Up(3)= Up(3)/d

      dotVpix = Up(1)*viewVec(1)+Up(2)*viewVec(2)+Up(3)*viewVec(3)

C---  compute the zenith angles for the view and sun vectors 

      if (dotVpix .LE. 1.0 .AND. dotVpix .GE. -1.0) then
        zaview = RAD2DEG* acos((-1.0)*dotVpix)
      else
        if (dotVpix .GT. 1.0 .AND. dotVpix .LT. 1.001) then
           zaview = 180.0
        else if (dotVpix .GT. (-1.001) .AND. dotVpix .LT. (-1.0)) then
             zaview = 0.0
        else
             zaview = BUGDOUBLE
        endif 
      endif


      UpDotSun = Up(1)*sunVec(1)+Up(2)*sunVec(2)+Up(3)*sunVec(3)
      
C      print *,'sunVec:',sunVec(1),sunVec(2),sunVec(3)
C      print *,'pixVec:',pixVec(1),pixVec(2),pixVec(3)
C      print *,'satVec:',satVec(1),satVec(2),satVec(3)
C      print *,'Up:',Up(1),Up(2),Up(3)
C      print *,'UpDotSun:',UpDotSun 

      if (UpDotSun .LE. 1.0 .AND. UpDotSun .GE. -1.0) then
         zasun = RAD2DEG* acos(UpDotSun)
      else
         if (UpDotSun .GT. 1.0 .AND. UpDotSun .LT. 1.001) then
            zasun = 0.0
         else if (UpDotSun .GT. (-1.001) .AND. UpDotSun .LT. -1.0) then
            zasun = 180.0
         else
            zasun = BUGDOUBLE
         endif
      endif 


C-----If pixel in view
      if (zaview .LT. 90.0) then
C-------If sun shines on pixel
        if (zasun .LT. 90.0) then
C---------------------------------------------------------------
C------    get relected view in ECR/ECEF 
C---------------------------------------------------------------
           viewRefl(1) = viewVec(1) - 2*dotVpix*Up(1)
           viewRefl(2) = viewVec(2) - 2*dotVpix*Up(2)
           viewRefl(3) = viewVec(3) - 2*dotVpix*Up(3)
           d=sqrt(viewRefl(1)*viewRefl(1)+viewRefl(2)*viewRefl(2)+
     &       viewRefl(3)*viewRefl(3))
           viewRefl(1)=viewRefl(1)/d
           viewRefl(2)=viewRefl(2)/d
           viewRefl(3)=viewRefl(3)/d

C---------------------------------------------------------------
C------    get anglular separation from Sun vector
C---------------------------------------------------------------
           sunDotView = sunVec(1)*viewRefl(1)+sunVec(2)*viewRefl(2)+
     &                  sunVec(3)*viewRefl(3)        

           if (sunDotView .LE. 1.0 .AND. sunDotView .GE. -1.0) then
              sunGlintAng = RAD2DEG*acos(sunDotView)
           else
              if (sunDotView .GT. 1.0 .AND. sunDotView .LT. 1.001) then
                 sunGlintAng = 0.0
              else if (sunDotView .GT. (-1.001) .AND. sunDotView .LT. 
     &                 (-1.0)) then
                   sunGlintAng = 180.0
              else
                   sunGlintAng = BUGDOUBLE
              endif
           endif

C---------------------------------------------------------------
C------    get sun_view_azim angle
C---------------------------------------------------------------
C----------get corss-view vector: Cvec = viewVec X Up
           Cvec(1)=viewVec(2)*Up(3)-viewVec(3)*Up(2)
           Cvec(2)=viewVec(3)*Up(1)-viewVec(1)*Up(3)
           Cvec(3)=viewVec(1)*Up(2)-viewVec(2)*Up(1)
           d=sqrt(Cvec(1)*Cvec(1)+Cvec(2)*Cvec(2)+Cvec(3)*Cvec(3))          

C----------if view is not straight down
           if (d .GT. 0.000001) then
C-------------unitize Cvec[]
              Cvec(1) =Cvec(1)/d
              Cvec(2) =Cvec(2)/d
              Cvec(3) =Cvec(3)/d
C-------------get along-view vector: Avec = Up X Cvec
              Avec(1)=Up(2)*Cvec(3)-Up(3)*Cvec(2)
              Avec(2)=Up(3)*Cvec(1)-Up(1)*Cvec(3)
              Avec(3)=Up(1)*Cvec(2)-Up(2)*Cvec(1)
              d=sqrt(Avec(1)*Avec(1)+Avec(2)*Avec(2)+Avec(3)*Avec(3))
              Avec(1) =Avec(1)/d
              Avec(2) =Avec(2)/d
              Avec(3) =Avec(3)/d
C-------------compute sun_view_azim angle
              sunDotA=sunVec(1)*Avec(1)+sunVec(2)*Avec(2)+
     &                sunVec(3)*Avec(3)
              sunDotC=sunVec(1)*Cvec(1)+sunVec(2)*Cvec(2)+
     &                sunVec(3)*Cvec(3)
              sun_view_az = RAD2DEG*atan2((-1.0)*sunDotC, sunDotA)
           else
C-------------satellite look straight down
              sun_view_az = 0.0
           endif 
C-------sun not on pixel
        else
           sunGlintAng  = MISSDOUBLE
           sun_view_az = MISSDOUBLE
        endif
C-----pixel not in view
      else
         sunGlintAng  = MISSDOUBLE
         sun_view_az = MISSDOUBLE
      endif

C-----return the sunGlintAng
      getSunGlintAng=sunGlintAng 

      return
      end
