C*
C*-----------------------------------------------------------------------------
C* SUBROUTINE getSunVecECR()
C*
C* Description:
C*   This function computes the unit sun position vector in ECR
C*   for the given time.
C*
C* Input parameters:
C*   year          int     - 4 digit year
C*   doy           int     - day of year (1-366)
C*   secs_of_day   real*4  - seconds of the day including fractional part
C*   GHA           double  - Greenwich hour angle (rad)
C*
C* Output:
C*   sunVec(3)     double  - unit sun position vector in ECR
C*
C* Functions called:
C*   sun2000SG()
C*   convGCI2ECR()
C*
C*
C* Log:
C*   10/25/2006        Joyce Chou        first created
C*
C*-----------------------------------------------------------------------------
C23456789012345678901234567890123456789012345678901234567890123456789012
      SUBROUTINE getSunVecECR(year,doy,secs_of_day,GHA,sunVec)

      implicit none

C---  input parameters
      integer*4 year,doy
      real*4    secs_of_day
      double precision GHA

C---  output parameter
      double precision sunVec(3)

      double precision rs,mag      

C---  get sunVec() in GCI coordinate
      CALL sun2000SG(year,doy,DBLE(secs_of_day),sunVec,rs)

C---  convert sunVec[] from GCI to ECR coordinate
      CALL convGCI2ECR(sunVec,GHA)
 
C---  unitize the vector 
      mag=sqrt(sunVec(1)*sunVec(1)+sunVec(2)*sunVec(2)+sunVec(3)*sunVec(
     &3))
      sunVec(1)=sunVec(1)/mag
      sunVec(2)=sunVec(2)/mag
      sunVec(3)=sunVec(3)/mag

      RETURN
      END

C*******************************************************************
C* SUBROUTINE convGCI2ECR()
C*
C* Description:
C*   This function converts a vector from GCI/ECI coordinates
C*   to ECR/ECEF coordinates
C*
C* Input parameters:
C*   Vec(3)        double  - the vector in GCI
C*   GHA           double  - Greenwich hour angle (rad)
C*
C* Output:
C*   Vec(3)        double  - the vector in ECR
C*
C* Functions called: none
C*
C*
C* Log:
C*   10/25/2006        Joyce Chou        first created
C*
C*******************************************************************/

      SUBROUTINE convGCI2ECR(Vec,GHA)

      implicit none
      double precision Vec(3),GHA

      double precision cGHA,sGHA,nVEC(3)

      cGHA=cos(GHA)
      sGHA=sin(GHA)

C---  convert from GCI to ECR coordinates by applying a     
C---  rotation angle of GHA.                               
C---  note the Z component (satVec(3))  does not change   

      nVEC(1) =Vec(1)*cGHA+Vec(2)*sGHA
      nVEC(2) =(-1.0)*Vec(1)*sGHA+Vec(2)*cGHA

C---  assign the vector back to Vec[] 
      Vec(1)=nVEC(1)
      Vec(2)=nVEC(2)

      RETURN 
      END

 
