C***************************************************************
C* FUNCTION date2jdaySG()
C*
C* Description:
C*   Function to convert calendar date to julian day (day of year).
C*
C* Input:
C*   year   4 digit year
C*   month  month of the year
C*   day    day of the month
C*
C* Return:
C*   doy   int   julian day   (day of year)
C*
C* Log:
C*   10/25/2006   Joyce Chou    first created
C*
C****************************************************************
C23456789012345678901234567890123456789012345678901234567890123456789012

      integer*4 FUNCTION date2jdaySG(year,month,day)
       
      implicit none

C-----input parameters
      integer*4 year,month,day
 
      integer*4 accumDays(12)
      integer*4 leapAccumDays(12)
      DATA accumDays /0,31,59,90,120,151,181,212,243,273,304,334/
      DATA leapAccumDays /0,31,60,91,121,152,182,213,244,274,305,335/

      integer*4 jld
      integer*4 M4,M100,M400


      M4=MOD(year,4) 
      M100=MOD(year,100) 
      M400=MOD(year,400) 

C-----if it is a leap year
      IF((M4 .EQ. 0 .AND. M100 .NE. 0) .OR. (M400 .EQ. 0)) then
         date2jdaySG=leapAccumDays(month)+day
C-----if it is not a leap year
      ELSE
         date2jdaySG=accumDays(month)+day
      ENDIF

      RETURN
      END


C***************************************************************
C* FUNCTION getDOY()
C*
C* Description:
C*   Function to convert calendar date to julian day (day of year).
C*   also get the seconds of the day.
C*
C* Input:
C*   year   int    - 4 digit year
C*   month  int    - month of the year
C*   day    int    - day of the month
C*   hour   int    - hour
C*   min    int    - minute
C*   fsec   real*4 - second in float
C*
C* Output:
C*   secs_of_day  real*4  - seconds of the day
C*
C* Return:
C*   doy           int    - julian day   (day of year)
C*
C* Routines called:
C*   date2jdaySG()
C*
C* Log:
C*   10/25/2006   Joyce Chou    first created
C*
C****************************************************************

C23456789012345678901234567890123456789012345678901234567890123456789012
      integer*4 FUNCTION getDOY(year,month,day,hour,min,fsec,secs_of_day
     &) 

      implicit none

C-----input parameters            
      integer*4 year,month,day,hour,min
      real*4    fsec 

C-----output parameter            
      real*4    secs_of_day 

C-----declare function
      integer*4 date2jdaySG 
      
C-----compute the seconds of the day
      secs_of_day = hour*3600.0 + min*60.0 + fsec

C-----get the day od year (julian day)
      getDOY=date2jdaySG(year,month,day)

      RETURN
      END

       
C****************************************************************************/
C****************************************************************************/
C* NOTE            NOTE            NOTE              NOTE                   */
C* The following functions are taken directly from the GEOLOCATION package. */
C* But the names have been changed to avoid double definition               */
C*                                                                          */
C****************************************************************************/
C****************************************************************************/

     
C***************************************************************************
C*
C*  SUBROUTINE gha2000SG()
C*
C*  Description:
C*    This subroutine computes the Greenwich hour angle in degrees for the
C*    input time. It uses the model referenced in the Astronomical Almanac
C*    for 1984, Section S (Supplement) and documented for the SeaWiFS
C*    Project in "Constants and Parameters for SeaWiFS Mission Operations",
C*    in TBD.  It includes the correction to mean sideral time for nutation
C*    as well as precession.
C*
C*    This routine originally coded in FORTRAN by Fred Patt, GSC, Laurel, MD.
C*
C*  Input Parameters:
C*    iyr       int        year (four digits)
C*    day       double     day (time of day as fraction)
C*
C*  Output Parameters:
C*    gha       double     Greenwich hour angle (rad)
C*
C*
C*  Routines called:
C*    jdSG        computes day of year from calendar date
C*    ephparmSG   computes mean solar longitude and anomaly and
C*                mean lunar longitude and ascending node
C*    nutateSG    compute nutation corrections to longitude and obliquity
C*
C*
C**************************************************************************
 
      SUBROUTINE gha2000SG(iyr,day,gha)
 
      implicit none
       
C-----input parameters
      integer*4 iyr
      double precision day,gha
  
      integer*4 imon
      DATA imon /1/
      integer*4   nutime,nt
      DATA nutime /-99999/
      double precision    fday,ip,t,gmst
      double precision    xls,gs,xlm,omega,dpsi,eps
      integer*4 iday,jday
      double precision RAD2DEG,DEG2RAD
      parameter (RAD2DEG=57.29577951,DEG2RAD=0.017453292) 

      integer*4 jdSG 
   
       
C---  Compute days since J2000 
      iday=day
      fday=day-iday
      jday=jdSG(iyr,imon,iday)
      t=jday-2451545.50+fday


C---  Compute Greenwich Mean Sidereal Time (degrees) 
      gmst=100.46061840 + 0.98564736630*t + 2.908e-13 *t*t

C---  Check if need to compute nutation correction for this day 
C      nt=t
C      IF (nt .NE. nutime) THEN
C          nutime=nt
          CALL ephparmsSG(t,xls,gs,xlm,omega)
          CALL nutateSG(t,xls,gs,xlm,omega,dpsi,eps)
C      ENDIF

C---  include apparent time correction and time-of-day 
      gha = gmst + dpsi*cos(eps/RAD2DEG) +fday*360.0
      gha = DMOD(gha,360.0)

      if (gha .LT. 0.0) gha=gha+360.0

C---  convert gha from deg to rad 
      gha = gha*DEG2RAD

      RETURN
      END 



C***************************************************************************
C*
C*  SUBROUTINE sun2000SG()
C*
C*  Description:
C*    This subroutine computes the sun vector in geocentric inertial
C*    (equatorial) coordinates. It uses the model referenced in The
C*    Astonomical Almanac for 1984, Section S (Supplement) and
C*    documented for the SeaWiFS Project in "Constants and Parameters
C*    for SeaWiFS Mission Operations", in TBD. The accuracy of the Sun
C*    vector is approximately 0.1 arcminute. This routine originally
C*    coded in FORTRAN by Fred Patt, GSC, Laurel, MD.
C*
C*  Input Parameters:
C*    iyr     int      year, four digits (i.e., 1993)
C*    iday    int      day of year (1-366)
C*    sec     double   seconds of day
C*
C*  Output Parameters:
C*    sun(3)  double    unit sun vector in geocentric inertial
C*                      coordinates of date
C*    rs      double    magnitude of the sun vector (AU)
C*
C*
C*  Routines called:
C*    jdSG           computes Julian day from calendar date
C*    ephparmsSG     computes mean solar longitude and anomaly and
C*                   mean lunar longitude and ascending node
C*    nutateSG       compute nutation corrections to longitude
C*                   and obliquity
C*
C***************************************************************************

      SUBROUTINE sun2000SG(iyr,iday,sec,sun,rs)

      implicit none
      
C-----input parameters
      integer*4 iyr,iday
      double precision    sec

C-----output parameters
      double precision    sun(3),rs
       
      double precision RAD2DEG,DEG2RAD,XK
      parameter (RAD2DEG=57.29577951,DEG2RAD=0.017453292) 
      parameter (XK=0.0056932) 
      integer*4 imon
      DATA imon /1/
        

      double precision dls,t,xls,gs,xlm,omega,dpsi,eps,g2,g4,g5
      double precision xlsg,xlsa
      integer*4 nutime,nt
      DATA nutime /-99999/
      double precision sgs, cgs, sg2, cg2, sg5, cg5

      integer*4 jdSG



C---  Compute floating point days since Jan 1.5, 2000. 
C---  Note that the Julian day starts at noon on the specified date 
      t=jdSG(iyr,imon,iday)-2451545.00+(sec-43200.0)/86400.0

C---  Compute solar ephemeris parameters 
      CALL ephparmsSG(t,xls,gs,xlm,omega)

C---  Check if need to compute nutation corrections for this day 
C      nt=t
C      if (nt .NE. nutime) THEN
C         nutime=nt
         CALL nutateSG(t,xls,gs,xlm,omega,dpsi,eps)
C      endif


C---  Compute planet mean anomalies. 
C---  Venus mean anomaly 
      g2=50.40828+1.60213022*t
      g2=dmod(g2,360.0)

C---  Mars mean anomaly 
      g4=19.38816+0.52402078*t
      g4=dmod(g4,360.0)

C---  Jupiter mean anomaly 
      g5=20.35116+0.08309121*t
      g5=dmod(g5,360.0)

      sgs=sin(gs/RAD2DEG)
      cgs=cos(gs/RAD2DEG)
      sg2=sin(g2/RAD2DEG)
      cg2=cos(g2/RAD2DEG)
      sg5=sin(g5/RAD2DEG)
      cg5=cos(g5/RAD2DEG)

C---  Compute solar distance (AU) 
      rs = 1.00014-0.01671*cgs - 0.00014*(1.-2 * sgs * sgs)

C---  Compute geometric solar longitude 
      dls=(6893.-4.6543463-4*t)*sgs
     &+72.*2*sgs*cgs
     &-7*(cgs*cg5+sgs*sg5)
     &+6.*sin((xlm-xls)/RAD2DEG)
     &+5.*sin((4.*gs-8.*g4+3.*g5)/RAD2DEG)
     &-5.* (1-2*(sgs*cg2-cgs*sg2)*(sgs*cg2-cgs*sg2))
     &-4.*(sgs*cg2-cgs*sg2)
     &+4.*cos((4.*gs-8.*g4+3.*g5)/RAD2DEG)
     &+3.*2*(sgs*cg2-cgs*sg2)*(cgs*cg2+sgs*sg2)
     &-3.*sg5
     &-3.*2*(sgs*cg5-cgs*sg5)*(cgs*cg5+sgs*sg5)

      xlsg=xls+dls/3600.0

C---  Compute apparent solar longitude includes corrections for
C---  nutation in longitude and velocity aberration.           
      xlsa=xlsg+dpsi-XK/(rs)

C---  Compute unit sun vector 

      sun(1)=cos(xlsa/RAD2DEG)
      sun(2)=sin(xlsa/RAD2DEG)*cos(eps/RAD2DEG)
      sun(3)=sin(xlsa/RAD2DEG)*sin(eps/RAD2DEG)

      RETURN
      END


C***************************************************************************
C*
C*  FUNCTION jdSG()
C*
C*  Description:
C*    This function converts a calendar date to the corresponding Julian
C*    day starting at noon on the calendar date. The algorithm used is
C*    from Van Flandern and Pulkkinen, Ap. J. Supplement Series 41,
C*    November 1079, p.400.
C*
C*  Input Parameters:
C*  i     int          4 digit year-e.g. 1970
C*  j     int          month (1-12)
C*  k     int          day (1-31)
C*
C*  Output Parameters:
C*  jd    int          Julian day
C*
C*  Called By:     sun2000SG
C*
C*  Routines called:
C*  none
C*
C***************************************************************************
      integer*4 FUNCTION jdSG(i,j,k) 
      implicit none

C-----input parameters
      integer*4 i,j,k

C---  compute the Julian day
      jdSG=367*i - 7*(i+(j+9)/12)/4 + 275*j/9 + k +1721014

C---  This additional calculation is needed only for dates outside of 
C---  the period March 1, 1900 to February 28, 2100.                 

C---  jdSG=jdSG + 15 - 3*((i+(j-9)/7)/100+1)/4


      RETURN
      END


C***************************************************************************
C*
C*  SUBROUTINE nutateSG()
C*
C*  Description:
C*  This subroutine computes the nutation in longitude and the obliquity
C*  of the ecliptic corrected for nutation.  It uses the model referenced
C*  in The Astronomical Alkmanac for 1984, Section S (Supplement) and
C*  documented for the SeaWiFS Project in "Constants and Parameters for
C*  SeaWiFS Mission Operations", in TBD.  These parameters are used to
C*  compute the apparent time correction to the Greenwich Hour Angle and
C*  for the calculation of the geocentric Sun vector.  The input
C*  ephemeris parameters are computed using subroutine ephparms.  Terms
C*  are included to 01 arcsecond.
C*
C*  Input Parameters:
C*  t     double  time indays since January 1, 2000 at 12 hours UT
C*  xls   double  mean solar longitude (degrees)
C*  gs    double  mean solar anomaly (degrees)
C*  xlm   double  mean lunar longitude (degrees)
C*  omega double  ascending node of mean lunar orbit (degrees)
C*
C*  Output Parameters:
C*  dpsi  double  nutation in longitude (degrees)
C*  eps   double  obliquity of the ecliptic (degrees)
C*
C*  Called By:     sun2000SG,gha2000SG
C*
C*  Routines called: None.
C*
C***************************************************************************

      SUBROUTINE nutateSG(t,xls,gs,xlm,omega,dpsi,eps)

      implicit none

C---  input parameters      
      double precision t,xls,gs,xlm,omega

C---  output parameters      
      double precision dpsi,eps

      double precision deps,epsm
      double precision RAD2DEG,DEG2RAD
      parameter (RAD2DEG=57.29577951,DEG2RAD=0.017453292) 

C---  Nutation in longitude 
      dpsi= -17.1996*sin(omega/RAD2DEG)+0.2062*sin(2.*omega/RAD2DEG)
     &      -1.3187*sin(2.*xls/RAD2DEG)+0.1426*sin(gs/RAD2DEG)
     &      -0.2274*sin(2.*xlm/RAD2DEG)

C---  Mean obliquity of the Ecliptic 
      epsm=23.4392910 - 3.560e-7 *t

C---  Nutation in obliquity 
      deps=9.2025*cos(omega/RAD2DEG)+0.5736*cos(2.*xls/RAD2DEG)

C---  True obliquity of the ecliptic 
      eps=epsm+deps/3600.0

      dpsi=dpsi/3600.0

      RETURN
      END



C***************************************************************************
C*
C*  SUBROUTINE ephparmsSG()
C*
C*  Description:
C*  This subroutine computes ephemeris parameters used by other Mission
C*  Operations routines: the solar mean longitude and mean anomaly, and
C*  the lunar mean longitude and mean ascending node.  It uses the model
C*  referenced in The Astronomical Almanac for 1984, Section S (Supplement)
C*  and documented for the SeaWiFS Project in "Constants and Parameters for
C*  SeaWiFS Mission Operations", in TBD.  These parameters are used to
C*  compute the solar longitude and the nutation in longitude and obliquity.
C*
C*  Input Parameters:
C*  t       double   time in days since January 1, 2000 at 12 hours UT
C*
C*  Output Parameters:
C*  xls     double   mean solar longitude (degrees)
C*  gs      double   mean solar anomaly (degrees)
C*  xlm     double   mean lunar longitude (degrees)
C*  omega   double   ascending node of mean lunar orbit (degrees)
C*
C*
C*  Called By:      sun2000SG,gha2000SG
C*
C*  Routines called: None.
C***************************************************************************


      SUBROUTINE ephparmsSG(t,xls,gs,xlm,omega)
      implicit none
     
C---  input parameter
      double precision t

C---  output parameters
      double precision xls,gs,xlm,omega

C---  sun mean longitude 
      xls=280.465920 + 0.98564735160*t
      xls=DMOD(xls,360.0)

C---  sun mean anomaly 
      gs=357.527720 + 0.98560028310*t
      gs=DMOD(gs,360.0)

C---  moon mean longitude 
      xlm=218.316430+13.176396480*t
      xlm=DMOD(xlm,360.0)

C---  ascending node of moon's mean orbit 
      omega=125.044520 - 0.05295376480*t
      omega=DMOD(omega,360.0)

      return
      END
